#!/usr/bin/env python3
# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.

"""Multi-view test a video classification model."""

import numpy as np
import slowfast.utils.checkpoint as cu
import slowfast.utils.distributed as du
import slowfast.utils.logging as logging
import torch
import torch.distributed as dist
import torch.nn as nn
from slowfast.datasets import loader
from slowfast.models import build_model
import math
import slowfast.utils.distributed as du

logger = logging.get_logger(__name__)



class TestMeter(object):
    def __init__(self, num_videos):
        self.num_videos = num_videos
        self.num_clips = 30
        self.video_preds = torch.zeros((num_videos, 400)).cuda()
        self.audio_preds = torch.zeros((num_videos, 400)).cuda()
        self._labels = torch.zeros(num_videos).cuda()

    def update_states(self, audio_preds, video_preds, labels, clip_ids):
        for index in range(audio_preds.shape[0]):
            vid_id = int(clip_ids[index]) // self.num_clips
            self._labels[vid_id] = labels[index]
            self.video_preds[vid_id] += video_preds[index]
            self.audio_preds[vid_id] += audio_preds[index]

    def print_final(self):
        _, video_preds = torch.max(self.video_preds, dim=1)
        _, audio_preds = torch.max(self.audio_preds, dim=1)
        print("video top1:", (video_preds == self._labels).sum() / self.num_videos)
        print("audio top1:", (audio_preds == self._labels).sum() / self.num_videos)

def perform_test(test_loader, model, cfg):
    """
    For classification:
    Perform mutli-view testing that uniformly samples N clips from a video along
    its temporal axis. For each clip, it takes 3 crops to cover the spatial
    dimension, followed by averaging the softmax scores across all Nx3 views to
    form a video-level prediction. All video predictions are compared to
    ground-truth labels and the final testing performance is logged.
    For detection:
    Perform fully-convolutional testing on the full frames without crop.
    Args:
        test_loader (loader): video testing loader.
        model (model): the pretrained video model to test.
        test_meter (TestMeter): testing meters to log and ensemble the testing
            results.
        cfg (CfgNode): configs. Details can be found in
            slowfast/config/defaults.py
        writer (TensorboardWriter object, optional): TensorboardWriter object
            to writer Tensorboard log.
    """
    class Classifier(nn.Module):
        def __init__(self):
            super(Classifier, self).__init__()
            self.video_clf = nn.Linear(2048, 400)
            self.audio_clf = nn.Linear(2048, 400)

        def forward(self, audio, video):
            audio_hat = self.audio_clf(audio)
            video_hat = self.video_clf(video)
            return audio_hat, video_hat

    classifier = Classifier()
    path = "/code/slowfast/naive_linear.pt"
    classifier.load_state_dict({k.replace('module.', ''): v for k, v in torch.load(path, map_location="cpu").items()})
    classifier = classifier.cuda()

    # data_size = len(test_loader)
    test_meter = TestMeter(num_videos=19866)

    model.eval()
    running_video_correct = 0
    running_audio_correct = 0
    running_samples = 0

    for cur_iter, (audio, video, labels, video_idx, meta) in enumerate(test_loader):
        if cfg.NUM_GPUS:
                # Transfer the data to the current GPU device.
            if isinstance(video, (list,)):
                for i in range(len(video)):
                    video[i] = video[i].cuda(non_blocking=True)
            else:
                video = video.cuda(non_blocking=True)

            # Transfer the data to the current GPU device.
            labels = labels.cuda()
            audio = audio.cuda(non_blocking=True)
            video_idx = video_idx.cuda()
            # Perform the forward pass.
        with torch.no_grad():
            audio_f, video_f = model.module.get_features((audio, video))
            audio_hat, video_hat = classifier(audio_f, video_f)
            audio_hat = torch.nn.functional.softmax(audio_hat, dim=1)
            video_hat = torch.nn.functional.softmax(video_hat, dim=1)

        audio_hat, video_hat, labels, video_idx = du.all_gather([
            audio_hat, video_hat, labels, video_idx
        ])
        running_audio_correct += (audio_hat.argmax(1) == labels).sum().item()
        running_video_correct += (video_hat.argmax(1) == labels).sum().item()

        running_samples += audio_hat.shape[0]
        print("sample:{}, running_a_acc:{}, running_v_acc:{}".format(running_samples, running_audio_correct/running_samples, running_video_correct/running_samples))

        test_meter.update_states(audio_hat, video_hat, labels, video_idx)
    test_meter.print_final()




def test(cfg):
    """
    Perform multi-view testing on the pretrained video model.
    Args:
        cfg (CfgNode): configs. Details can be found in
            slowfast/config/defaults.py
    """
    # Set up environment.
    du.init_distributed_training(cfg)
    # Set random seed from configs.
    np.random.seed(cfg.RNG_SEED)
    torch.manual_seed(cfg.RNG_SEED)

    # Setup logging format.
    logging.setup_logging(cfg.OUTPUT_DIR)

    # Print config.
    logger.info("Test with config:")
    logger.info(cfg)

    # Build the video model and print model statistics.
    model = build_model(cfg)

    # if du.is_master_proc() and cfg.LOG_MODEL_INFO:
    #     misc.log_model_info(model, cfg, use_train_input=False)

    cu.load_test_checkpoint(cfg, model)

    # Create video testing loaders.
    test_loader = loader.construct_loader(cfg, "test")
    logger.info("train model for {} iterations".format(len(test_loader)))


    # # Perform multi-view test on the entire dataset.
    test_meter = perform_test(test_loader, model, cfg=cfg)

